<?php
// Start the session
session_start();

// Check if the user is logged in, if not then redirect to login page
if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
    header("location: ../login.php");
    exit;
}

// Check if the user is an employee, if not then redirect to admin dashboard
if ($_SESSION["role"] !== "employee") {
    header("location: ../admin/dashboard.php");
    exit;
}

// Include config file
require_once "../config/db.php";

// Check existence of id parameter before processing further
if (isset($_GET["id"]) && !empty(trim($_GET["id"]))) {
    // Get URL parameter
    $id = trim($_GET["id"]);

    // Prepare a select statement
    $sql = "SELECT t.*, u.username AS assigned_to_username, a.username AS assigned_by_username, t.attachment_path
            FROM tasks t
            JOIN users u ON t.assigned_to = u.user_id
            JOIN users a ON t.assigned_by = a.user_id
            WHERE t.task_id = ? AND t.assigned_to = ?";
    if ($stmt = mysqli_prepare($link, $sql)) {
        // Bind variables to the prepared statement as parameters
        mysqli_stmt_bind_param($stmt, "ii", $param_id, $param_assigned_to);

        // Set parameters
        $param_id = $id;
        $param_assigned_to = $_SESSION["id"];

        // Attempt to execute the prepared statement
        if (mysqli_stmt_execute($stmt)) {
            $result = mysqli_stmt_get_result($stmt);

            if (mysqli_num_rows($result) == 1) {
                /* Fetch result row as an associative array. Since the result set
                contains only one row, we don't need to use a while loop */
                $task = mysqli_fetch_assoc($result);
            } else {
                // URL doesn't contain valid id or task not assigned to this employee. Redirect to error page
                header("location: ../error.php");
                exit();
            }
        } else {
            echo "Oops! Something went wrong. Please try again later.";
        }
    }

    // Close statement
    mysqli_stmt_close($stmt);

    // Fetch comments for the task
    $comments = [];
    $sql_comments = "SELECT c.*, u.username FROM comments c JOIN users u ON c.user_id = u.user_id WHERE c.task_id = ? ORDER BY c.created_at ASC";
    if ($stmt_comments = mysqli_prepare($link, $sql_comments)) {
        mysqli_stmt_bind_param($stmt_comments, "i", $param_task_id);
        $param_task_id = $id;
        if (mysqli_stmt_execute($stmt_comments)) {
            $result_comments = mysqli_stmt_get_result($stmt_comments);
            while ($row_comment = mysqli_fetch_assoc($result_comments)) {
                $comments[] = $row_comment;
            }
        }
        mysqli_stmt_close($stmt_comments);
    }

    // Close connection
    mysqli_close($link);
} else {
    // URL doesn't contain id parameter. Redirect to error page
    header("location: ../error.php");
    exit();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>View Task</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f4f7f6;
            color: #333;
        }
        .wrapper {
            display: flex;
            min-height: 100vh;
            transition: all 0.3s ease;
        }
        .sidebar {
            width: 250px;
            background-color: #2c3e50;
            color: #ecf0f1;
            padding: 20px;
            box-shadow: 2px 0 5px rgba(0,0,0,0.1);
            display: flex;
            flex-direction: column;
            transition: all 0.3s ease;
        }
        .sidebar h3 {
            text-align: center;
            margin-bottom: 30px;
            color: #ecf0f1;
        }
        .sidebar ul {
            list-style: none;
            padding: 0;
            flex-grow: 1;
        }
        .sidebar ul li {
            margin-bottom: 15px;
        }
        .sidebar ul li a {
            color: #ecf0f1;
            text-decoration: none;
            padding: 10px 15px;
            display: block;
            border-radius: 5px;
            transition: background-color 0.3s ease;
        }
        .sidebar ul li a:hover,
        .sidebar ul li a.active {
            background-color: #34495e;
        }
        .main-content {
            flex-grow: 1;
            padding: 30px;
            transition: all 0.3s ease;
        }
        .navbar {
            background-color: #ffffff;
            padding: 15px 30px;
            border-bottom: 1px solid #ddd;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        }
        .navbar h2 {
            margin: 0;
            color: #333;
        }
        .navbar .profile-info {
            display: flex;
            align-items: center;
        }
        .navbar .profile-info .username {
            margin-right: 15px;
            font-weight: bold;
            color: #555;
        }
        .navbar .btn-logout {
            background-color: #e74c3c;
            color: #fff;
            border: none;
            padding: 8px 15px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease;
        }
        .navbar .btn-logout:hover {
            background-color: #c0392b;
        }
        .card {
            background-color: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            margin-bottom: 30px;
            padding: 20px;
        }
        .card-header {
            font-size: 1.5rem;
            margin-bottom: 20px;
            color: #2c3e50;
            border-bottom: 1px solid #eee;
            padding-bottom: 10px;
        }
        .task-details p {
            margin-bottom: 10px;
        }
        .task-details strong {
            display: inline-block;
            width: 120px;
        }
        .comments-section {
            margin-top: 30px;
            border-top: 1px solid #eee;
            padding-top: 20px;
        }
        .comment {
            background-color: #f9f9f9;
            border: 1px solid #ddd;
            border-radius: 5px;
            padding: 10px;
            margin-bottom: 15px;
        }
        .comment-meta {
            font-size: 0.9em;
            color: #777;
            margin-bottom: 5px;
        }
        .comment-text {
            margin-bottom: 0;
        }
        .form-group {
            margin-bottom: 15px;
        }
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        .form-control {
            width: 100%;
            padding: 10px;
            border: 1px solid #ced4da;
            border-radius: 4px;
            box-sizing: border-box;
        }
        .btn-primary {
            background-color: #007bff;
            border-color: #007bff;
            color: #fff;
            padding: 10px 15px;
            border-radius: 4px;
            cursor: pointer;
        }
        .btn-primary:hover {
            background-color: #0056b3;
            border-color: #0056b3;
        }
        .btn-secondary {
            background-color: #6c757d;
            border-color: #6c757d;
            color: #fff;
            padding: 10px 15px;
            border-radius: 4px;
            cursor: pointer;
            margin-left: 10px;
        }
        .btn-secondary:hover {
            background-color: #5a6268;
            border-color: #545b62;
        }

        /* Responsive styles */
        @media (max-width: 768px) {
            .wrapper {
                flex-direction: column;
            }
            .sidebar {
                height: auto;
                padding: 15px;
                box-shadow: none;
                         box-shadow: none;
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    z-index: 999; 
            }
            .sidebar ul {
                display: none; /* Hidden by default */
                flex-direction: column;
            }
            .sidebar.active ul {
                display: flex; /* Show when active */
            }
            .sidebar .sidebar-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
            }
            .sidebar .sidebar-header h3 {
                margin-bottom: 0;
            }
            .main-content {
                padding: 15px;
            }
            .navbar {
                flex-direction: column;
                align-items: flex-start;
                padding: 15px;
            }
            .navbar h2 {
                margin-bottom: 10px;
            }
            .navbar .profile-info {
                margin-top: 10px;
                width: 100%;
                justify-content: space-between;
            }
            .navbar .profile-info .username {
                margin-right: 0;
            }
        }
    </style>
</head>
<body>
    <div class="wrapper" id="wrapper">
        <nav class="sidebar" id="sidebar">
            <div class="sidebar-header">
                               <h3>Work Allotment</h3>

                <button type="button" id="sidebarCollapse" class="btn btn-dark d-md-none">
                    <i class="bi bi-list"></i>
                </button>
            </div>
            <ul>
                <li><a href="dashboard.php"><i class="bi bi-grid-fill"></i> Dashboard</a></li>
                <li><a href="../logout.php"><i class="bi bi-box-arrow-right"></i> Logout</a></li>
            </ul>
        </nav>

         <div class="main-content">
            <br>
            <header class="navbar mt-5">
                <h2>View Task</h2>
                <div class="profile-info">
                    <span class="username">Welcome, <?php echo htmlspecialchars($_SESSION["username"]); ?></span>
                    <a href="../logout.php" class="btn-logout">Logout</a>
                </div>
            </header>

            <div class="container-fluid mt-4">
                <div class="card">
                    <div class="card-header">Task Details</div>
                    <div class="card-body task-details">
                        <p><strong>Title:</strong> <?php echo htmlspecialchars($task['title']); ?></p>
                        <p><strong>Description:</strong> <?php echo nl2br(htmlspecialchars($task['description'])); ?></p>
                        <p><strong>Assigned To:</strong> <?php echo htmlspecialchars($task['assigned_to_username']); ?></p>
                        <p><strong>Assigned By:</strong> <?php echo htmlspecialchars($task['assigned_by_username']); ?></p>
                        <p><strong>Deadline:</strong> <?php echo htmlspecialchars($task['deadline']); ?></p>
                        <p><strong>Priority:</strong> <?php echo htmlspecialchars($task['priority']); ?></p>
                        <p><strong>Status:</strong> <?php echo htmlspecialchars($task['status']); ?></p>
                        <p><strong>Created At:</strong> <?php echo htmlspecialchars($task['created_at']); ?></p>
                        <p><strong>Last Updated:</strong> <?php echo htmlspecialchars($task['updated_at']); ?></p>
                        <?php if (!empty($task['attachment_path'])): ?>
                            <p><strong>Attachment:</strong> <a href="../admin/<?php echo htmlspecialchars($task['attachment_path']); ?>" target="_blank">View Attachment</a></p>
                        <?php endif; ?>
                        <a href="dashboard.php" class="btn btn-secondary">Back to Tasks</a>
                    </div>
                </div>

                <div class="card comments-section">
                    <div class="card-header">Comments</div>
                    <div class="card-body">
                        <div id="comments-list">
                            <?php if (!empty($comments)): ?>
                                <?php foreach ($comments as $comment): ?>
                                    <div class="comment">
                                        <div class="comment-meta">
                                            <strong><?php echo htmlspecialchars($comment['username']); ?></strong> on <?php echo htmlspecialchars($comment['created_at']); ?>
                                        </div>
                                        <p class="comment-text"><?php echo nl2br(htmlspecialchars($comment['comment_text'])); ?></p>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <p>No comments yet.</p>
                            <?php endif; ?>
                        </div>

                        <hr>
                        <h4>Add a Comment</h4>
                        <form id="comment-form">
                            <div class="form-group">
                                <label for="comment_text">Your Comment</label>
                                <textarea name="comment_text" id="comment_text" class="form-control" rows="3" required></textarea>
                            </div>
                            <input type="hidden" name="task_id" value="<?php echo $task['task_id']; ?>">
                            <input type="hidden" name="user_id" value="<?php echo $_SESSION['id']; ?>">
                            <button type="submit" class="btn btn-primary">Submit Comment</button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script>
        $(document).ready(function() {
            $('#comment-form').on('submit', function(e) {
                e.preventDefault();
                var formData = $(this).serialize();

                $.ajax({
                    type: 'POST',
                    url: '../api/add_comment.php', // API endpoint for adding comments
                    data: formData,
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            Swal.fire(
                                'Success!',
                                'Comment added successfully.',
                                'success'
                            ).then(() => {
                                // Append new comment to the list
                                var newCommentHtml = `
                                    <div class="comment">
                                        <div class="comment-meta">
                                            <strong>${response.username}</strong> on ${response.created_at}
                                        </div>
                                        <p class="comment-text">${response.comment_text}</p>
                                    </div>
                                `;
                                $('#comments-list').append(newCommentHtml);
                                $('#comment_text').val(''); // Clear textarea
                            });
                        } else {
                            Swal.fire(
                                'Error!',
                                response.message,
                                'error'
                            );
                        }
                    },
                    error: function(xhr, status, error) {
                        Swal.fire(
                            'Error!',
                            'An error occurred while submitting your comment.',
                            'error'
                        );
                        console.error(xhr.responseText);
                    }
                });
            });
        });

        // Sidebar toggle for mobile
        document.getElementById('sidebarCollapse').addEventListener('click', function() {
            document.getElementById('sidebar').classList.toggle('active');
        });
    </script>
</body>
</html>